package ftpserver

import (
	"bufio"
	"errors"
	"fmt"
	"io"
	"net"
	"strings"
	"sync"
	"time"

	log "github.com/fclairamb/go-log"
)

// HASHAlgo is the enumerable that represents the supported HASH algorithms.
type HASHAlgo int8

// Supported hash algorithms
const (
	HASHAlgoCRC32 HASHAlgo = iota
	HASHAlgoMD5
	HASHAlgoSHA1
	HASHAlgoSHA256
	HASHAlgoSHA512
)

// TransferType is the enumerable that represents the supported transfer types.
type TransferType int8

// Supported transfer type
const (
	TransferTypeASCII TransferType = iota
	TransferTypeBinary
)

// DataChannel is the enumerable that represents the data channel (active or passive)
type DataChannel int8

// Supported data channel types
const (
	DataChannelPassive DataChannel = iota + 1
	DataChannelActive
)

const (
	maxCommandSize = 4096
)

var (
	errNoTransferConnection  = errors.New("unable to open transfer: no transfer connection")
	errTLSRequired           = errors.New("unable to open transfer: TLS is required")
	errInvalidTLSRequirement = errors.New("invalid TLS requirement")
)

func getHashMapping() map[string]HASHAlgo {
	mapping := make(map[string]HASHAlgo)
	mapping["CRC32"] = HASHAlgoCRC32
	mapping["MD5"] = HASHAlgoMD5
	mapping["SHA-1"] = HASHAlgoSHA1
	mapping["SHA-256"] = HASHAlgoSHA256
	mapping["SHA-512"] = HASHAlgoSHA512

	return mapping
}

func getHashName(algo HASHAlgo) string {
	hashName := ""
	hashMapping := getHashMapping()

	for k, v := range hashMapping {
		if v == algo {
			hashName = k
		}
	}

	return hashName
}

type clientHandler struct {
	connectedAt         time.Time       // Date of connection
	paramsMutex         sync.RWMutex    // mutex to protect the parameters exposed to the library users
	driver              ClientDriver    // Client handling driver
	conn                net.Conn        // TCP connection
	user                string          // Authenticated user
	path                string          // Current path
	listPath            string          // Path for NLST/LIST requests
	clnt                string          // Identified client
	command             string          // Command received on the connection
	ctxRnfr             string          // Rename from
	logger              log.Logger      // Client handler logging
	transferWg          sync.WaitGroup  // wait group for command that open a transfer connection
	transfer            transferHandler // Transfer connection (passive or active)s
	extra               any             // Additional application-specific data
	server              *FtpServer      // Server on which the connection was accepted
	writer              *bufio.Writer   // Writer on the TCP connection
	reader              *bufio.Reader   // Reader on the TCP connection
	ctxRest             int64           // Restart point
	transferMu          sync.Mutex      // this mutex will protect the transfer parameters
	id                  uint32          // ID of the client
	selectedHashAlgo    HASHAlgo        // algorithm used when we receive the HASH command
	currentTransferType TransferType    // current transfer type
	lastDataChannel     DataChannel     // Last data channel mode (passive or active)
	debug               bool            // Show debugging info on the server side
	transferTLS         bool            // Use TLS for transfer connection
	controlTLS          bool            // Use TLS for control connection
	isTransferOpen      bool            // indicate if the transfer connection is opened
	isTransferAborted   bool            // indicate if the transfer was aborted
	connClosed          bool            // indicates if the connection has been commanded to close
	tlsRequirement      TLSRequirement  // TLS requirement to respect
}

// newClientHandler initializes a client handler when someone connects
func (server *FtpServer) newClientHandler(
	connection net.Conn,
	clientID uint32,
	transferType TransferType,
) *clientHandler {
	return &clientHandler{
		server:              server,
		conn:                connection,
		id:                  clientID,
		writer:              bufio.NewWriter(connection),
		reader:              bufio.NewReaderSize(connection, maxCommandSize),
		connectedAt:         time.Now().UTC(),
		path:                "/",
		selectedHashAlgo:    HASHAlgoSHA256,
		currentTransferType: transferType,
		logger:              server.Logger.With("clientId", clientID),
	}
}

// disconnects the connection without any other messaging
func (c *clientHandler) disconnect() error {
	if c.connClosed {
		return nil
	}

	err := c.conn.Close()
	if err != nil {
		err = newNetworkError("error closing control connection", err)
		c.logger.Warn(
			"Problem disconnecting a client",
			"err", err,
		)
	}

	c.connClosed = true

	return err
}

// Path provides the current working directory of the client
func (c *clientHandler) Path() string {
	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.path
}

// SetPath changes the current working directory
func (c *clientHandler) SetPath(value string) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.path = value
}

// getListPath returns the path for the last LIST/NLST request
func (c *clientHandler) getListPath() string {
	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.listPath
}

// SetListPath changes the path for the last LIST/NLST request
func (c *clientHandler) SetListPath(value string) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.listPath = value
}

// Debug defines if we will list all interaction
func (c *clientHandler) Debug() bool {
	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.debug
}

// SetDebug changes the debug flag
func (c *clientHandler) SetDebug(debug bool) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.debug = debug
}

// ID provides the client's ID
func (c *clientHandler) ID() uint32 {
	return c.id
}

// RemoteAddr returns the remote network address.
func (c *clientHandler) RemoteAddr() net.Addr {
	return c.conn.RemoteAddr()
}

// LocalAddr returns the local network address.
func (c *clientHandler) LocalAddr() net.Addr {
	return c.conn.LocalAddr()
}

// GetClientVersion returns the identified client, can be empty.
func (c *clientHandler) GetClientVersion() string {
	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.clnt
}

func (c *clientHandler) setClientVersion(value string) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.clnt = value
}

// HasTLSForControl returns true if the control connection is over TLS
func (c *clientHandler) HasTLSForControl() bool {
	if c.server.settings.TLSRequired == ImplicitEncryption {
		return true
	}

	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.controlTLS
}

func (c *clientHandler) setTLSForControl(value bool) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.controlTLS = value
}

// HasTLSForTransfers returns true if the transfer connection is over TLS
func (c *clientHandler) HasTLSForTransfers() bool {
	if c.server.settings.TLSRequired == ImplicitEncryption {
		return true
	}

	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.transferTLS
}

func (c *clientHandler) SetExtra(extra any) {
	c.extra = extra
}

func (c *clientHandler) Extra() any {
	return c.extra
}

func (c *clientHandler) setTLSForTransfer(value bool) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.transferTLS = value
}

// SetTLSRequirement sets the TLS requirement to respect for this connection
func (c *clientHandler) SetTLSRequirement(requirement TLSRequirement) error {
	if requirement < ClearOrEncrypted || requirement > MandatoryEncryption {
		return errInvalidTLSRequirement
	}

	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.tlsRequirement = requirement

	return nil
}

func (c *clientHandler) isTLSRequired() bool {
	if c.server.settings.TLSRequired == MandatoryEncryption {
		return true
	}

	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.tlsRequirement == MandatoryEncryption
}

// GetLastCommand returns the last received command
func (c *clientHandler) GetLastCommand() string {
	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.command
}

// GetLastDataChannel returns the last data channel mode
func (c *clientHandler) GetLastDataChannel() DataChannel {
	c.paramsMutex.RLock()
	defer c.paramsMutex.RUnlock()

	return c.lastDataChannel
}

func (c *clientHandler) setLastCommand(cmd string) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.command = cmd
}

func (c *clientHandler) setLastDataChannel(channel DataChannel) {
	c.paramsMutex.Lock()
	defer c.paramsMutex.Unlock()

	c.lastDataChannel = channel
}

func (c *clientHandler) closeTransfer() error {
	var err error
	if c.transfer == nil {
		return nil
	}

	err = c.transfer.Close()
	c.isTransferOpen = false
	c.transfer = nil

	if c.debug {
		c.logger.Debug("Transfer connection closed")
	}

	if err != nil {
		err = fmt.Errorf("error closing transfer connection: %w", err)

		return err
	}

	return nil
}

// Close closes the active transfer, if any, and the control connection
func (c *clientHandler) Close() error {
	c.transferMu.Lock()
	defer c.transferMu.Unlock()

	// set isTransferAborted to true so any transfer in progress will not try to write
	// to the closed connection on transfer close
	c.isTransferAborted = true

	if err := c.closeTransfer(); err != nil {
		c.logger.Warn(
			"Problem closing a transfer on external close request",
			"err", err,
		)
	}

	// don't be tempted to send a message to the client before
	// closing the connection:
	//
	// 1) it is racy, we need to lock writeMessage to do this
	// 2) the client could wait for another response and so we break the protocol
	//
	// closing the connection from a different goroutine should be safe
	return c.disconnect()
}

// disconnects client and ends transfer notifying the driver
func (c *clientHandler) end() {
	c.server.driver.ClientDisconnected(c)
	c.server.clientDeparture(c)

	c.transferMu.Lock()
	if err := c.closeTransfer(); err != nil {
		c.logger.Warn(
			"Problem closing a transfer",
			"err", err,
		)
	}
	c.transferMu.Unlock()

	if err := c.disconnect(); err != nil {
		c.logger.Warn(
			"Problem disconnecting client on end",
			"err", err,
		)
	}
}

func (c *clientHandler) isCommandAborted() bool {
	c.transferMu.Lock()
	defer c.transferMu.Unlock()

	return c.isTransferAborted
}

// HandleCommands reads the stream of commands
func (c *clientHandler) HandleCommands() {
	defer c.end()

	if msg, err := c.server.driver.ClientConnected(c); err == nil {
		c.writeMessage(StatusServiceReady, msg)
	} else {
		c.writeMessage(StatusSyntaxErrorNotRecognised, msg)

		return
	}

	for {
		if c.readCommand() {
			return
		}
	}
}

func (c *clientHandler) readCommand() bool {
	if c.reader == nil {
		if c.debug {
			c.logger.Debug("Client disconnected", "clean", true)
		}

		return true
	}

	// florent(2018-01-14): #58: IDLE timeout: Preparing the deadline before we read
	if c.server.settings.IdleTimeout > 0 {
		if err := c.conn.SetDeadline(
			time.Now().Add(time.Duration(time.Second.Nanoseconds() * int64(c.server.settings.IdleTimeout)))); err != nil {
			c.logger.Error("Network error", "err", err)
		}
	}

	lineSlice, isPrefix, err := c.reader.ReadLine()

	if isPrefix {
		if c.debug {
			c.logger.Warn("Received line too long, disconnecting client",
				"size", len(lineSlice))
		}

		return true
	}

	if err != nil {
		c.handleCommandsStreamError(err)

		return true
	}

	line := string(lineSlice)

	if c.debug {
		c.logger.Debug("Received line", "line", line)
	}

	c.handleCommand(line)

	return false
}

func (c *clientHandler) handleCommandsStreamError(err error) {
	// florent(2018-01-14): #58: IDLE timeout: Adding some code to deal with the deadline
	var errNetError net.Error
	if errors.As(err, &errNetError) { //nolint:nestif // too much effort to change for now
		if errNetError.Timeout() {
			// We have to extend the deadline now
			if errSet := c.conn.SetDeadline(time.Now().Add(time.Minute)); errSet != nil {
				c.logger.Error("Could not set read deadline", "err", errSet)
			}

			c.logger.Info("Client IDLE timeout", "err", err)
			c.writeMessage(
				StatusServiceNotAvailable,
				fmt.Sprintf("command timeout (%d seconds): closing control connection", c.server.settings.IdleTimeout))

			if errFlush := c.writer.Flush(); errFlush != nil {
				c.logger.Error("Flush error", "err", errFlush)
			}

			return
		}

		c.logger.Error("Network error", "err", err)
	} else {
		if errors.Is(err, io.EOF) {
			if c.debug {
				c.logger.Debug("Client disconnected", "clean", false)
			}
		} else {
			c.logger.Error("Read error", "err", err)
		}
	}
}

// handleCommand takes care of executing the received line
func (c *clientHandler) handleCommand(line string) {
	command, param := parseLine(line)
	command = strings.ToUpper(command)

	cmdDesc := commandsMap[command]
	if cmdDesc == nil {
		// Search among commands having a "special semantic". They
		// should be sent by following the RFC-959 procedure of sending
		// Telnet IP/Synch sequence (chr 242 and 255) as OOB data but
		// since many ftp clients don't do it correctly we check the
		// command suffix.
		for _, cmd := range specialAttentionCommands {
			if strings.HasSuffix(command, cmd) {
				cmdDesc = commandsMap[cmd]
				command = cmd

				break
			}
		}

		if cmdDesc == nil {
			c.logger.Warn("Unknown command", "command", command)
			c.setLastCommand(command)
			c.writeMessage(StatusSyntaxErrorNotRecognised, fmt.Sprintf("Unknown command %#v", command))

			return
		}
	}

	if c.driver == nil && !cmdDesc.Open {
		c.writeMessage(StatusNotLoggedIn, "Please login with USER and PASS")

		return
	}

	// All commands are serialized except the ones that require special action.
	// Special action commands are not executed in a separate goroutine so we can
	// have at most one command that can open a transfer connection and one special
	// action command running at the same time.
	// Only server STAT is a special action command so we do an additional check here
	if !cmdDesc.SpecialAction || (command == "STAT" && param != "") {
		c.transferWg.Wait()
	}

	c.setLastCommand(command)

	if cmdDesc.TransferRelated {
		// these commands will be started in a separate goroutine so
		// they can be aborted.
		// We cannot have two concurrent transfers so also set isTransferAborted
		// to false here.
		// isTransferAborted could remain to true if the previous command is
		// aborted and it does not open a transfer connection, see "transferFile"
		// for details. For this to happen a client should send an ABOR before
		// receiving the StatusFileStatusOK response. This is very unlikely
		// A lock is not required here, we cannot have another concurrent ABOR
		// or transfer active here
		c.isTransferAborted = false

		c.transferWg.Add(1)

		go func(cmd, param string) {
			defer c.transferWg.Done()

			c.executeCommandFn(cmdDesc, cmd, param)
		}(command, param)
	} else {
		c.executeCommandFn(cmdDesc, command, param)
	}
}

func (c *clientHandler) executeCommandFn(cmdDesc *CommandDescription, command, param string) {
	// Let's prepare to recover in case there's a command error
	defer func() {
		if r := recover(); r != nil {
			c.writeMessage(StatusSyntaxErrorNotRecognised, fmt.Sprintf("Unhandled internal error: %s", r))
			c.logger.Warn(
				"Internal command handling error",
				"err", r,
				"command", command,
				"param", param,
			)
		}
	}()

	if err := cmdDesc.Fn(c, param); err != nil {
		c.writeMessage(StatusSyntaxErrorNotRecognised, fmt.Sprintf("Error: %s", err))
	}
}

func (c *clientHandler) writeLine(line string) {
	if c.debug {
		c.logger.Debug("Sending answer", "line", line)
	}

	if _, err := fmt.Fprintf(c.writer, "%s\r\n", line); err != nil {
		c.logger.Warn(
			"Answer couldn't be sent",
			"line", line,
			"err", err,
		)
	}

	if err := c.writer.Flush(); err != nil {
		c.logger.Warn(
			"Couldn't flush line",
			"err", err,
		)
	}
}

func (c *clientHandler) writeMessage(code int, message string) {
	lines := getMessageLines(message)

	for idx, line := range lines {
		if idx < len(lines)-1 {
			c.writeLine(fmt.Sprintf("%d-%s", code, line))
		} else {
			c.writeLine(fmt.Sprintf("%d %s", code, line))
		}
	}
}

func (c *clientHandler) GetTranferInfo() string {
	if c.transfer == nil {
		return ""
	}

	return c.transfer.GetInfo()
}

func (c *clientHandler) TransferOpen(info string) (net.Conn, error) {
	c.transferMu.Lock()
	defer c.transferMu.Unlock()

	if c.transfer == nil {
		// a transfer could be aborted before it is opened, in this case no response should be returned
		if c.isTransferAborted {
			c.isTransferAborted = false

			return nil, errNoTransferConnection
		}

		c.writeMessage(StatusActionNotTaken, errNoTransferConnection.Error())

		return nil, errNoTransferConnection
	}

	if c.isTLSRequired() && !c.HasTLSForTransfers() {
		c.writeMessage(StatusServiceNotAvailable, errTLSRequired.Error())

		return nil, errTLSRequired
	}

	conn, err := c.transfer.Open()
	if err != nil {
		c.logger.Warn(
			"Unable to open transfer",
			"error", err)

		c.writeMessage(StatusCannotOpenDataConnection, err.Error())

		err = newNetworkError("Unable to open transfer", err)

		return nil, err
	}

	c.isTransferOpen = true
	c.transfer.SetInfo(info)

	c.writeMessage(StatusFileStatusOK, "Using transfer connection")

	if c.debug {
		c.logger.Debug(
			"Transfer connection opened",
			"remoteAddr", conn.RemoteAddr().String(),
			"localAddr", conn.LocalAddr().String())
	}

	return conn, nil
}

func (c *clientHandler) TransferClose(err error) {
	c.transferMu.Lock()
	defer c.transferMu.Unlock()

	errClose := c.closeTransfer()
	if errClose != nil {
		c.logger.Warn(
			"Problem closing transfer connection",
			"err", err,
		)
	}

	// if the transfer was aborted we don't have to send a response
	if c.isTransferAborted {
		c.isTransferAborted = false

		return
	}

	switch {
	case err == nil && errClose == nil:
		c.writeMessage(StatusClosingDataConn, "Closing transfer connection")
	case errClose != nil:
		c.writeMessage(StatusActionNotTaken, fmt.Sprintf("Issue during transfer close: %v", errClose))
	case err != nil:
		c.writeMessage(getErrorCode(err, StatusActionNotTaken), fmt.Sprintf("Issue during transfer: %v", err))
	}
}

func (c *clientHandler) checkDataConnectionRequirement(dataConnIP net.IP, channelType DataChannel) error {
	var requirement DataConnectionRequirement

	switch channelType {
	case DataChannelActive:
		requirement = c.server.settings.ActiveConnectionsCheck
	case DataChannelPassive:
		requirement = c.server.settings.PasvConnectionsCheck
	}

	switch requirement {
	case IPMatchRequired:
		controlConnIP, err := getIPFromRemoteAddr(c.RemoteAddr())
		if err != nil {
			return err
		}

		if !controlConnIP.Equal(dataConnIP) {
			return &ipValidationError{error: fmt.Sprintf("data connection ip address %v "+
				"does not match control connection ip address %v",
				dataConnIP, controlConnIP)}
		}

		return nil
	case IPMatchDisabled:
		return nil
	default:
		return &ipValidationError{error: fmt.Sprintf("unhandled data connection requirement: %v",
			requirement)}
	}
}

func getIPFromRemoteAddr(remoteAddr net.Addr) (net.IP, error) {
	if remoteAddr == nil {
		return nil, &ipValidationError{error: "nil remote address"}
	}

	ipAddress, _, err := net.SplitHostPort(remoteAddr.String())
	if err != nil {
		return nil, fmt.Errorf("error parsing remote address: %w", err)
	}

	remoteIP := net.ParseIP(ipAddress)
	if remoteIP == nil {
		return nil, &ipValidationError{error: fmt.Sprintf("invalid remote IP: %v", ipAddress)}
	}

	return remoteIP, nil
}

func parseLine(line string) (string, string) {
	params := strings.SplitN(line, " ", 2)
	if len(params) == 1 {
		return params[0], ""
	}

	return params[0], params[1]
}

func (c *clientHandler) multilineAnswer(code int, message string) func() {
	c.writeLine(fmt.Sprintf("%d-%s", code, message))

	return func() {
		c.writeLine(fmt.Sprintf("%d End", code))
	}
}

func getMessageLines(message string) []string {
	lines := make([]string, 0, 1)
	sc := bufio.NewScanner(strings.NewReader(message))

	for sc.Scan() {
		lines = append(lines, sc.Text())
	}

	if len(lines) == 0 {
		lines = append(lines, "")
	}

	return lines
}
